<?php

namespace App\Repositories;

use App\Models\BaseModel;
use App\Models\User;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Repository
{
    protected null|BaseModel|Model $model = null;
    private int $paginationLimit = 20;

    public static function startTransaction()
    {
        DB::beginTransaction();
    }

    public static function commitTransaction()
    {
        DB::commit();
    }

    public static function rollbackTransaction()
    {
        DB::rollBack();
    }

    public function all(mixed $select = '*')
    {
        return $this->getModel()->select($select)->get();
    }

    /**
     * @return mixed
     */
    public function getModel()
    {
        return $this->model;
    }

    /**
     * @param mixed $model
     */
    public function setModel(BaseModel|Model $model): void
    {
        $this->model = $model;
    }

    public function create($data): ?BaseModel
    {

        return $this->getModel()->create($data);
    }

    public function updateOrCreate($unique, $data)
    {

        $created = $this->getModel()->updateOrCreate($unique, $data);
        if ($created) {
            return $created;
        }
        return false;
    }

    public function findBy($filed, $value, $checkEventId = false)
    {
        return $this->getModel()->where($filed, $value)
            ->when($checkEventId, function ($query) {
                $query->where('event_id', request('event')['id']);
            })
            ->first();
    }

    public function findIn($listOfIds)
    {
        $record = $this->getModel()->whereIn('id', $listOfIds)->get();
        if (!$record)
            return false;
        return $record;
    }

    public function findWith($id, $with)
    {
        $record = $this->getModel()->with($with)->find($id);
        if (!$record)
            return false;
        return $record;
    }

    public function find($id): ?BaseModel
    {
        return $this->getModel()->find($id);
    }

    public function deleteIn($ids, array $where = null): bool
    {
        $record = (is_null($where)) ? $this->getModel()->whereIn('id', $ids) : $this->getModel()->whereIn('id', $ids)->where($where[0], $where[1]);
        if (!$record)
            return false;
        return $record->delete();
    }

    public function delete($id): bool
    {
        $record = $this->getModel()->find($id);

        if (!$record)
            return false;

        return $record->delete();
    }

    public function forceDeleteIn($ids, array $where = null): bool
    {
        $record = (is_null($where)) ? $this->getModel()->whereIn('id', $ids) : $this->getModel()->whereIn('id', $ids)->where($where[0], $where[1]);
        if (!$record)
            return false;
        return $record->forceDelete();
    }

    public function forceDelete($id): bool
    {
        $record = $this->getModel()->find($id);
        if (!$record)
            return false;

        return $record->forceDelete();
    }

    public function deleteWhere($column, $value)
    {
        $record = $this->getModel()->where($column, $value)->first();
        if (!$record)
            return false;
        return $record->delete();
    }

    public function updateWhere($where, $data): bool
    {
        $record = $this->getModel()->where($where[0], $where[1])->first();
        if (!$record)
            return false;

        return $record->update($data);
    }

    /**
     * @param $id
     * @param $data
     * @return BaseModel|User|null
     */
    public function update($id, $data): BaseModel|User|null
    {
        $record = $this->getModel()->find($id);

        if (!$record)
            return null;

        if (!$record->update($data))
            return null;

        return $record->refresh();
    }

    public function getWhere($column, $value, $checkActiveStatus = false): Collection|array
    {
        return $this->getModel()->where($column, $value)->when($checkActiveStatus, function ($query){
            $query->active();
        })->get();
    }

    /**
     * @return int
     */
    public function getPaginationLimit(): int
    {
        return $this->paginationLimit;
    }

    /**
     * @param int $paginationLimit
     */
    public function setPaginationLimit(int $paginationLimit): void
    {
        $this->paginationLimit = $paginationLimit;
    }
}

